# frozen_string_literal: true

require 'tmpdir'

require_relative 'test_helper'
require 'haichi'

describe Haichi::ConfigBase do
  describe '#config class definition' do
    it 'allows defining a simple configuration' do
      _(create_class do
          default_path '/etc/haichi/config.yml'
          field :name, String
          field :duration, Integer, default: 10_000
          field :local_dirs, Array, subtype: String, default: [Dir.home]
        end)
    end

    it 'allows defining a configuration with multiple default paths' do
      _(cls = create_class do
          default_path '/etc/haichi/config.yml'
          default_path '/usr/lib/haichi/default.yml'
          field :name, String
        end)

      _(cls.default_paths.length).must_equal 2
    end

    it 'keeps the order of multiple default paths as defined' do
      _(cls = create_class do
          default_path '/etc/haichi/config.yml'
          default_path '/usr/lib/haichi/default.yml'
          field :name, String
        end)

      _(cls.default_paths).must_equal %w[/etc/haichi/config.yml /usr/lib/haichi/default.yml]
    end

    it 'rejects default values of the wrong type' do
      _ do
        create_class do
          field :duration, Integer, default: '500'
        end
      end.must_raise TypeError
    end

    it 'rejects arguments for "allowed" that are not of the correct type' do
      _ do
        create_class do
          field :duration, Integer, allowed: %w[0 50 100]
        end
      end.must_raise TypeError
    end

    it 'only allows "nil" or an Array for "allowed"' do
      _ do
        create_class do
          field :duration, Integer, default: 500, allowed: 'foo'
        end
      end.must_raise TypeError
    end

    it 'rejects default values not in allowed types' do
      _ do
        create_class do
          field :duration, Integer, default: 500, allowed: [0, 50, 100]
        end
      end.must_raise Haichi::ConfigError
    end

    it 'has a working bool workaround' do
      _ do
        create_class do
          field :should_use, Haichi::Bool, default: '500'
        end
      end.must_raise TypeError

      _ do
        create_class do
          field :should_use, Haichi::Bool, default: false
        end
      end

      _ do
        create_class do
          field :should_use, Haichi::Bool, default: true
        end
      end
    end

    it 'rejects default values of the wrong type including subtypes' do
      _ do
        create_class do
          field :names, Array, subtype: String, default: [500]
        end
      end.must_raise TypeError
    end
  end

  describe '#config class loading' do
    before do
      @tmp_conf_dir = Dir.mktmpdir 'haichi_spec'
      conf_path = File.join @tmp_conf_dir, 'config.yml'
      @conf_path = conf_path
      @cls = create_class do
        default_path conf_path
        field :name, String, default: 'foo'
        field :duration, Integer
        field :all_mods, Array, default: [], rename: 'all-mods'
        field :exec, Haichi::Bool, default: true
      end
    end

    after do
      FileUtils.rm_r @tmp_conf_dir
    end

    it 'can create a new config object from a hash' do
      _(conf = @cls.new(name: 'foo', duration: 42))
      _(conf.name).must_equal 'foo'
      _(conf.duration).must_equal 42
      _(conf.exec).must_equal true
    end

    it 'rejects configs with missing entries' do
      h = { 'name' => 'bar', 'all-mods' => %w[qoo qux] }
      _ do
        @cls.new(**h)
      end.must_raise Haichi::ConfigError
    end

    it 'correctly renames fields' do
      h = { 'name' => 'bar', 'duration' => 84, 'all-mods' => %w[qoo qux] }
      _(conf = @cls.new(**h))
      _(conf.all_mods).must_equal %w[qoo qux]
    end

    it 'can create a new config object from a YAMl string' do
      y = "---\nname: 'bar'\nduration: 84\nexec: false"
      _(conf = @cls.from_yaml(y))
      _(conf.name).must_equal 'bar'
      _(conf.duration).must_equal 84
      _(conf.exec).must_equal false
    end

    it 'raises a ConfigError when parsing YAML fails' do
      y = "---\nname: 'bar'\nduration: 84\nexec: false\n'qoo'"
      _ do
        @cls.from_yaml(y)
      end.must_raise Haichi::ConfigError
    end

    it 'can create a new config object from a file' do
      File.open @conf_path, 'w' do |file|
        file.write "---\nname: 'bar'\nduration: 84"
      end

      _(conf = @cls.from_file(@conf_path))
      _(conf.name).must_equal 'bar'
      _(conf.duration).must_equal 84
      _(conf.exec).must_equal true
    end

    it 'can create a new config object from a default path' do
      File.open @conf_path, 'w' do |file|
        file.write "---\nname: 'bar'\nduration: 84"
      end

      _(@cls.default_paths).wont_be_empty
      _(conf = @cls.load)
      _(conf.name).must_equal 'bar'
      _(conf.duration).must_equal 84
      _(conf.exec).must_equal true
    end

    it 'rejects empty keys when they are not optional' do
      y = "---\nname: 'bar'\nduration:\n"
      _ do
        @cls.from_yaml(y)
      end.must_raise TypeError
    end

    it 'correctly sets default values' do
      conf = @cls.new duration: 42
      _(conf.name).must_equal 'foo'
      _(conf.exec).must_equal true
    end

    it 'allows accessing the fields' do
      conf = @cls.new(name: 'foo', duration: 42)
      _(conf.name).must_equal 'foo'
      _(conf.duration).must_equal 42
    end

    it 'overwrites a default value with one explicitly specified' do
      conf = @cls.new name: 'bar', duration: 42
      _(conf.name).must_equal 'bar'
    end
  end
end
