# frozen_string_literal: true

require 'yaml'
require 'set'

module Haichi
  class ConfigError < StandardError; end

  # Base class to derive from for all your config needs.
  class ConfigBase
    def initialize(**kwargs)
      kwargs = Haichi.apply_renames(self.class.field_configs, kwargs).then { |r| Haichi.symbolise_hash r }
      check_missing(**kwargs)
      assign(**kwargs)
      validate_types
      postprocess
    end

    def all_fields
      puts self.class.field_configs.keys
    end

    private

    def postprocess; end

    FieldConfig = Struct.new 'FieldConfig', :type, :subtype, :default, :optional, :rename, :allowed
    LoadConfig = Struct.new 'LoadConfig', :paths

    def check_missing(**kwargs)
      missing_keys = self.class.field_configs.reject { |_, v| v.optional || !v.default.nil? }
                         .map { |k, _| k }.to_set - kwargs.keys.to_set
      raise ConfigError, "Missing required config keys #{missing_keys}" unless missing_keys.empty?
    end

    def validate_types
      ckey_set = self.class.field_configs.keys.to_set

      # Type check the values
      ckey_set.each do |key|
        fconf = self.class.field_configs[key]
        val = send key
        raise TypeError, "'#{key}' is required and must not be nil" if val.nil? && !fconf.optional && fconf.default.nil?
        unless (val.nil? && fconf.optional) || val.is_a?(fconf.type)
          raise TypeError, "'#{key}' is not of type #{fconf.type}"
        end

        raise ConfigError, "'#{val}' is not in allowed types" if !fconf.allowed.nil? && !fconf.allowed.include?(val)

        val_class = val.class
        if val_class == Array
          next if val.empty? || fconf.subtype.nil?

          raise TypeError, "Subtype for '#{key}' should be #{fconf.subtype}" unless val.all? do |v|
                                                                                      v.is_a? fconf.subtype
                                                                                    end
        elsif val_class == Hash
          next if val.empty? || fconf.subtype.nil?

          raise TypeError, "Subtype for '#{key}' should be #{fconf.subtype}" unless val.values.all? do |v|
                                                                                      v.is_a? fconf.subtype
                                                                                    end
        end
      end
    end

    # Assign the values contained in the dict to their respective instance variables
    def assign(**kwargs)
      # Get a set of all defined config keys
      ckey_set = self.class.field_configs.keys.to_set
      # Get a set of all provided keys
      pkey_set = kwargs.keys.to_set
      # Get a set of missing keys to later fill in default values if they are available
      missing_keys = ckey_set - pkey_set

      # Go over all provided keys and values that have been given as kwargs and are defined
      # in the config and assign them
      accepted = kwargs.select { |k, _| ckey_set.include? k }.to_set
      accepted.each do |key, val|
        fconf = self.class.field_configs[key]
        key = "@#{key}".to_sym

        if fconf.type.is_a? ConfigBase
          instance_variable_set key, fconf.subtype.new(**val)
        elsif fconf.subtype&.superclass == ConfigBase && fconf.type == Array
          instance_variable_set(key, val.map { |v| fconf.subtype.new(**v) })
        else
          instance_variable_set key, val
        end
      end

      # Go over missing keys and fill them with default values, if available
      missing_keys.each do |key|
        dval = self.class.field_configs[key].default
        key = "@#{key}".to_sym
        instance_variable_set(key, dval) unless dval.nil?
      end
    end

    class << self
      attr_writer :field_configs, :load_config

      def field_configs
        @field_configs ||= {}
      end

      def load_config
        @load_config ||= LoadConfig.new([])
      end

      def default_paths
        load_config.paths
      end

      # Define a new field on the resulting config class.
      def field(name, type, subtype: nil, default: nil, optional: false, rename: nil, allowed: nil)
        name = name.to_sym
        raise TypeError, 'Invalid type for default' unless default.nil? || default.is_a?(type)
        raise ConfigError, "Redefining fields is not allowed: #{name}" if field_configs.include? name
        raise TypeError, 'Allowed needs to be an array of values or nil' unless allowed.nil? || allowed.is_a?(Array)

        check_subtype name, type, subtype, default unless subtype.nil? || default.nil?
        check_allowed type, allowed unless allowed.nil?
        if !allowed.nil? && !default.nil? && !allowed.include?(default)
          raise ConfigError, "'#{default}' is not in allowed values"
        end

        var_name = "@#{name}".to_sym
        field_configs[name] = FieldConfig.new type, subtype, default, optional, rename, allowed

        define_method name do
          instance_variable_get var_name
        end

        define_method "#{name}=" do |val|
          instance_variable_set var_name, val
        end
      end

      # Add a new default path to load the config from. Paths will be tried in the order they are defined in.
      def default_path(path)
        path = File.expand_path unless File.absolute_path? path
        load_config.paths << path
      end

      # Try to parse the provided YAML string and construct a config from that.
      def from_yaml(yaml_string)
        yaml_hash = YAML.safe_load yaml_string
        new(**yaml_hash)
      rescue YAML::Exception => e
        raise ConfigError, "could not parse YAML (#{e})"
      end

      # Try to read the config from the provided path.
      def from_file(path)
        path = File.expand_path path unless File.absolute_path? path
        File.open path, 'r' do |file|
          from_yaml file.read
        end
      rescue ConfigError => e
        raise ConfigError, "could not load '#{path}': #{e}"
      end

      # Try to load the config from one of the configured default paths.
      def load
        raise ConfigError, 'No default paths configured' if default_paths.empty?

        default_paths.each do |path|
          next unless File.exist? path

          return from_file path
        end
        raise ConfigError, 'Could not find any config file at the default paths'
      end

      private

      def check_subtype(name, type, subtype, default)
        ok = if type == Array
               default.empty? || default.all? { |e| e.is_a? subtype }
             elsif type == Hash
               default.empty? || default.values.all? { |e| e.is_a? subtype }
             else
               true
             end
        raise TypeError, "Subtype of element in default value does not match the one defined for '#{name}'" unless ok
      end

      def check_allowed(type, allowed)
        raise TypeError, "At least one value on 'allowed' is not of type '#{type}'" unless allowed.all? do |v|
                                                                                             v.is_a? type
                                                                                           end
      end
    end
  end

  def self.symbolise_hash(hash)
    res = {}
    hash.each { |k, v| res[k.to_sym] = v }
    res
  end

  def self.apply_renames(field_configs, hash)
    rename_map = {}
    res = {}
    field_configs.each.reject { |_, v| v.rename.nil? }.each { |k, v| rename_map[v.rename] = k }
    return hash if rename_map.empty?

    hash.each do |key, val|
      new_key = rename_map[key] || key
      res[new_key] = val
    end
    res
  end
end
