# frozen_string_literal: true

class BackupError < StandardError
end

# Formatting helpers for data sizes and durations
module Formatting
  def format_size(size, ext_spacing: false)
    size = Float size unless size.is_a? Float
    prefixes = %w[B KiB MiB GiB TiB]
    pidx = 0
    while size > 1024.0
      size /= 1024
      pidx += 1
    end
    if ext_spacing
      format('%<size>7.2f %<prefix>s', size: size, prefix: prefixes[pidx])
    else
      format('%<size>.2f %<prefix>s', size: size, prefix: prefixes[pidx])
    end
  end

  def format_dur(secs)
    return 'N/A' if secs.nil?

    secs = Float secs unless secs.is_a? Float
    hours = (secs / 3600).floor
    secs %= 3600
    mins = (secs / 60).floor
    secs %= 60
    format('%<hour>02dh %<minute>02dm %<second>02ds', hour: hours,
                                                      minute: mins,
                                                      second: secs)
  end

  def clear_line
    IO.console.goto_column 0
    IO.console.erase_line 2
  end
end

module RepoTools
  def format_path(repo)
    if repo.type == :ssh
      path = +''
      path << "#{repo.user}@" if repo.user
      path << "#{repo.host}:"
      path << repo.path
    else
      repo.path
    end
  end
end

module InitPaths
  DIST_DIR = File.join File.dirname(File.dirname(File.dirname(__FILE__))), 'dist'
  TEMPLATE_PATH = File.join DIST_DIR, 'config_template.yaml'
  SERVICE_PATH = File.join DIST_DIR, 'flood-backup.service'
  TIMER_PATH = File.join DIST_DIR, 'flood-backup.timer'
  SYSTEMD_TARGET_DIR_USER = File.join Dir.home, '.config/systemd/user/'
  SYSTEMD_TARGET_DIR_SYSTEM = '/etc/systemd/system/'
end

module Regexes
  RELOC = /at location (?<now>.*) was .* located at (?<prev>.*)\n/
end

class Progressbar
  include Formatting

  attr_reader :finished

  def initialize(label, current, maximum)
    @label = label
    @current = current
    @maximum = maximum
    @col_avail = IO.console.winsize[1].to_i
    @width = ((@col_avail - @label.length) * 0.65).floor
    @finished = false
  end

  def step(step_size: 1)
    @current += step_size
  end

  def set(value)
    @current = value
  end

  def finish
    @finished = true
    @current = @maximum
  end

  def finished?
    @finished
  end

  def active?
    !finished?
  end

  def draw
    filled = (@current.to_f / @maximum * @width).floor
    remaining = @width - filled
    percent = format('% 6.2f%%', @current.to_f / @maximum * 100)
    clear_line
    print "#{@label} ["
    print(':' * filled)
    print(' ' * remaining)
    print "] #{percent}"
    puts '' if @finished
  end

  def self.from_msgid(msgid, maximum)
    case msgid
    when 'repository.check'
      Progressbar.new 'Checking segments', 0, maximum
    when 'check.verify_data'
      Progressbar.new 'Verifying data', 0, maximum
    when 'repository.compact_segments'
      Progressbar.new 'Compacting segments', 0, maximum
    when 'check.rebuild_refcounts'
      Progressbar.new 'Checking archives', 0, maximum
    else
      logger.warn "Unknown msgid: #{msgid}"
      Progressbar.new 'Unknown operation', 0, maximum
    end
  end
end

def get_progressbar(msgid, maximum)
  case msgid
  when 'repository.check'
    Progressbar.new 'Checking segments', 0, maximum
  when 'check.verify_data'
    Progressbar.new 'Verifying data', 0, maximum
  when 'repository.compact_segments'
    Progressbar.new 'Compacting segments', 0, maximum
  when 'check.rebuild_refcounts'
    Progressbar.new 'Checking archives', 0, maximum
  else
    logger.warn "Unknown msgid: #{msgid}"
    Progressbar.new 'Unknown operation', 0, maximum
  end
end

def root_user?
  Process.uid.zero?
end

module FsUtils
  def get_approx_size(paths, excludes)
    paths.map! { |pa| File.expand_path pa }

    threads = paths.map do |path|
      Thread.new { get_size_recursive path, excludes }
    end
    threads.each(&:join)
    threads.map(&:value).sum
  end

  def get_size_recursive(root, excludes)
    return (File.size?(root) || 0) if File.file? root

    queue = [root]
    total_size = 0
    while (path = queue.shift)
      dir_entries = Dir.foreach(path)
                       .reject { |e| %w[. ..].include?(e) }
                       .map { |e| File.join path, e }
                       .reject { |e| excludes.any? { |ex| File.fnmatch ex, e } }

      dir_entries.each do |entry|
        if File.directory? entry
          queue << entry
          next
        elsif File.symlink? entry
          next
        end
        if (s = File.size? entry)
          total_size += s
        end
      end
    end
    total_size
  end

  def check_paths(paths)
    paths.reject do |path|
      base, = path.split ':'
      File.exist? File.expand_path(base)
    end
  end
end

def pairs(array)
  pairs = []
  (0..array.length - 2).each do |idx|
    pairs << array[idx..idx + 1]
  end
  pairs
end
