# frozen_string_literal: true

require 'flood/logging'
require 'flood/backup'
require 'flood/log_parsing'
require 'flood/utils'

class TaskRunner
  include Logging
  include RepoTools
  include Formatting
  include MsgParser

  def initialize(configs, opts = {})
    @configs = configs
    @opts = opts
    @config_names = @configs.map(&:name)
    logger.debug "Opts: #{opts}"
  end

  def backup(targets)
    # targets contains mappings 'conf_name => nil | [repo_name, ...]'
    logger.debug "Targets: #{targets}"
    valid_target_keys = targets.keys.intersection(config_names)
    logger.debug "Valid keys: #{valid_target_keys}"
    targets.select! { |t| valid_target_keys.include? t }
    raise BackupError, 'No valid target configs' if targets.empty?

    errors = []
    # TODO: store configs in Hash: name => config
    configs.filter { |c| targets.keys.include? c.name }.each do |conf|
      errors.concat single_backup(conf, targets[conf.name])
    end

    return if errors.empty?

    raise BackupError,
          "Could not execute all jobs successfully, got #{errors.length} error(s). Take a look at the logs"
  end

  def info(targets)
    target_repos = select_repos targets
    logger.debug "Targets: #{target_repos}"
    target_repos.each do |repo|
      borg = Borg.new(format_path(repo), repo.passphrase)
      info_hash = borg.info
      archives = borg.archives
      repo_info = parse_info info_hash
      puts "Info for '#{repo.name}' @ '#{repo.corresponding_backup}':"
      puts " Original size:      #{format_size repo_info[:tsize]}"
      puts " Compressed size:    #{format_size repo_info[:tcsize]}"
      puts " Deduplicated size:  #{format_size repo_info[:dsize]}"
      puts " Available archives: #{archives.length}"
      puts
    rescue BackupError => e
      logger.error "Could not get info for repo '#{repo.name}': #{e}"
    end
  end

  def archives(targets)
    logger.debug "Target repos: #{targets}"
    repos_to_query = select_repos targets

    logger.debug "Targets: #{repos_to_query}"

    repos_to_query.each do |repo|
      borg = Borg.new(format_path(repo), repo.passphrase)
      raise BackupError, "Repo #{repo.name} is not available" unless borg.repo_available? interactive: true

      archives = borg.archives
      logger.info "Archives for repo '#{repo.name}' @ '#{repo.corresponding_backup}'"
      puts ' ! No archives in this repo' if archives.empty?
      archives.each do |a|
        puts " - #{a}"
      end
      puts
    rescue BackupError => e
      logger.error "Could not list archives for repo '#{repo.name}': #{e}"
    end
  end

  def mount(config, repo, archive, mountpoint)
    raise BackupError, "'#{config}' is not a known config" unless config_names.include? config

    target_config = configs.filter { |c| c.name == config }.first
    raise BackupError, "'#{repo}' is not a known repo" unless target_config.repos.map(&:name).include? repo

    target_repo = target_config.repos.filter { |r| r.name == repo }.first
    borg = Borg.new(format_path(target_repo), target_repo.passphrase)
    raise BackupError, 'The target repo is not available' unless borg.repo_available? interactive: true

    unless File.exist? mountpoint
      print 'Mountpoint does not exist. Create it? (y/N) '
      ans = $stdin.gets.strip
      raise BackupError, 'Mountpoint not available' unless %w[y Y].include? ans

      Dir.mkdir mountpoint
    end

    raise BackupError, "'#{mountpoint}' is not a directory" unless File.directory? mountpoint

    borg.mount archive, mountpoint
    logger.info "Mount successful. Don't forget to unmount once you are done ('umount #{mountpoint}')."
  end

  def check(opts, targets)
    logger.debug 'Preparing check'
    repos_to_check = select_repos targets
    logger.debug "Selected repos: #{repos_to_check}"

    repos_to_check.each do |repo|
      logger.info "Executing check for repo #{repo.name} @ '#{repo.corresponding_backup}'"
      borg = Borg.new format_path(repo), repo.passphrase
      raise BackupError, 'The target repo is not available' unless borg.repo_available? interactive: true

      logger.debug "Created Borg instance: #{borg}"

      borg.check(verify_data: opts[:'verify-data']) do |queue|
        progress = nil
        while (elem = queue.pop)
          type, line = elem
          case type
          when :stdout
            logger.warn "Unexpected output: #{line}"
          when :stderr
            msg = parse_logmsg line
            if msg.is_a?(LogMessage)
              clear_line if progress&.active?
              level = msg.levelname.downcase
              logger.send Logging::METHOD_MAP[level], msg.message
              progress.draw if progress&.active?
            elsif msg.is_a?(ProgressPercent)
              progress = get_progressbar msg.msgid, msg.total if progress.nil? || progress.finished?
              progress.set msg.current unless msg.finished
              progress.finish if msg.finished
              progress.draw
            end
          end
        end
      end
      puts ''
    end
  end

  private

  attr_accessor :configs, :opts, :config_names

  def single_backup(config, target_repos)
    logger.info "Starting backup job for '#{config.name}'"
    logger.debug "This backup will target #{config.repos.length} repo(s)"
    local_conf = {}
    local_conf[:conf_name] = config.name
    local_conf[:paths] = config.paths
    local_conf[:excludes] = config.excludes
    local_conf[:pre_commands] = config.pre_commands
    local_conf[:background] = opts[:background]
    errors = []

    repos = if target_repos.nil?
              config.repos
            else
              config.repos.filter { |r| target_repos.include? r.name }
            end

    repos.each do |repo|
      logger.info "Target repo is '#{repo.name}'"
      BackupGen.new(local_conf, repo).run
      clear_env
      puts '' unless opts[:background]
    rescue BackupError => e
      clear_env
      errors << e
      logger.error "An error occurred during '#{config.name}::#{repo.name}': #{e}" unless opts[:'fail-fast']
      raise if opts[:'fail-fast']
    end
    errors
  end

  def select_repos(targets)
    invalid_keys = targets.keys - targets.keys.intersection(config_names)
    logger.debug "Invalid keys: #{invalid_keys}"
    invalid_keys.each do |k|
      logger.error "'#{k}' is not a known config"
      targets.delete k
    end
    raise BackupError, 'no valid targets' if targets.empty?

    configs.filter { |c| targets.keys.include? c.name }.map do |conf|
      if targets[conf.name].nil?
        conf.repos
      else
        conf.repos.filter { |r| targets[conf.name].include? r.name }
      end
    end.flatten
  end

  def parse_info(info_hash)
    out_hash = {}
    stats = info_hash['cache']['stats']
    out_hash[:tsize] = stats['total_size']
    out_hash[:tcsize] = stats['total_csize']
    out_hash[:dsize] = stats['unique_csize']

    out_hash
  end

  def clear_env
    ENV.delete 'BORG_REPO'
    ENV.delete 'BORG_PASSPHRASE'
  end
end
