# frozen_string_literal: true

require 'json'

class MsgError < StandardError
end

module MsgParser
  def parse_logmsg(msg)
    msg_json = JSON.parse msg.strip

    case msg_json['type']
    when 'archive_progress'
      ArchiveProgress.new msg_json
    when 'progress_message'
      ProgressMessage.new msg_json
    when 'progress_percent'
      ProgressPercent.new msg_json
    when 'log_message'
      LogMessage.new msg_json
    when 'question_prompt'
      InputPrompt.new msg_json
    when 'question_invalid_answer', 'question_accepted_false'
      PromptResult.new msg_json
    else
      raise MsgError, "unknown message #{msg_json}"
    end
  rescue JSON::ParserError
    Plaintext.new msg
  rescue StandardError => e
    puts "Something unexpected happened: #{e}"
    puts "Input: #{msg}"
    raise
  end
end

class ArchiveProgress
  attr_reader :nfiles, :osize, :csize, :dsize, :path, :time

  def initialize(msg)
    @osize = msg['original_size']
    @csize = msg['compressed_size']
    @dsize = msg['deduplicated_size']
    @nfiles = msg['nfiles']
    @path = msg['path']
    @time = msg['time'].to_f
  end
end

class ProgressMessage
  attr_reader :operation, :time, :msgid, :finished, :message

  def initialize(msg)
    @operation = msg['operation']
    @msgid = msg['msgid']
    @finished = msg['finished']
    @message = msg['message']
    @time = msg['time']
  end
end

class ProgressPercent
  attr_reader :operation, :time, :msgid, :finished, :message, :current, :info, :total

  def initialize(msg)
    @operation = msg['operation']
    @msgid = msg['msgid']
    @finished = msg['finished']
    @message = msg['message']
    @time = msg['time']
    @current = msg['current']
    @info = msg['info']
    @total = msg['total']
  end
end

class LogMessage
  attr_reader :time, :levelname, :name, :message, :msgid

  def initialize(msg)
    @time = msg['time']
    @levelname = msg['levelname']
    @name = msg['name']
    @message = msg['message']
    @msgid = msg['msgid']
  end
end

class InputPrompt
  attr_reader :message, :msgid

  def initialize(msg)
    @message = msg['message']
    @msgid = msg['msgid']
  end
end

class PromptResult
  attr_reader :msgid, :message, :prompt

  def initialize(msg)
    @msgid = msg['msgid']
    @message = msg['message']
    @prompt = msg['prompt']
  end
end

class Plaintext
  attr_reader :message

  def initialize(msg)
    @message = msg
  end
end
