# frozen_string_literal: true

require 'set'
require 'gli'
require 'flood/version'
require 'flood/configuration'
require 'flood/logging'
require 'flood/taskrunner'
require 'flood/utils'

BACKGROUND = false

class FloodCli
  extend Logging
  extend Formatting
  extend RepoTools
  extend GLI::App

  program_desc 'A simple wrapper for BorgBackup'
  version Flood::VERSION

  switch %i[v verbose], desc: 'Verbose output'

  pre do |opts, _, _, _|
    Logging.level = :debug if opts[:verbose]
    true
  end

  desc 'Create new backups'
  command :backup do |cmd|
    cmd.switch %i[b background],
               desc: 'Print reduced output more suited for a syslog'
    cmd.switch %i[fail-fast], desc: 'Immediately exit when encountering an error'
    cmd.action do |_global_opts, options, args|
      BACKGROUND = options[:background]
      configs = Configuration.load_configs
      raise BackupError, 'No valid configs found' if configs.empty?

      args = configs.filter(&:default_exec).map(&:name) if args.empty?
      raise BackupError, 'No configs selected' if args.empty?

      targets = parse_args args

      runner = TaskRunner.new configs, options
      runner.backup targets
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    end
  end

  desc 'List available configurations'
  command :configs do |cmd|
    cmd.action do |_, _, _, _|
      configs = Configuration.load_configs

      configs.each.with_index(1) do |conf, idx|
        header = if conf.default_exec
                   "#{idx} '#{conf.name}' (default) with repos"
                 else
                   "#{idx} '#{conf.name}' with repos"
                 end
        puts header
        conf.repos.each { |r| puts "  - #{r.name}" }
        puts
      end
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    end
  end

  desc 'List available archives'
  command :archives do |cmd|
    cmd.action do |_, _, args|
      configs = Configuration.load_configs
      logger.debug "Configs loaded: #{configs}"
      args = configs.filter(&:default_exec).map(&:name) if args.empty?
      raise BackupError, 'No configs selected' if args.empty?

      targets = parse_args args
      logger.debug "Targets: #{targets}"
      TaskRunner.new(configs).archives targets

    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    end
  end

  desc 'Print information about repos'
  command :info do |cmd|
    cmd.action do |_, _opts, args|
      configs = Configuration.load_configs
      args = configs.filter(&:default_exec).map(&:name) if args.empty?
      raise BackupError, 'No configs selected' if args.empty?

      targets = parse_args args
      TaskRunner.new(configs).info targets

    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    end
  end

  desc 'Mount a repo or an archive to access its contents'
  command :mount do |cmd|
    cmd.action do |_, _, args|
      raise BackupError, 'flood mount [target] [mountpoint]' if args.length < 2

      logger.warn 'Ignoring all arguments but the first two' if args.length > 2
      configs = Configuration.load_configs
      target = args.shift
      mountpoint = args.shift
      config, rest = target.split '::'
      raise BackupError, 'need a repo to mount' if rest.nil?

      repo, archive = rest.split '/'
      TaskRunner.new(configs).mount config, repo, archive, mountpoint

    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    end
  end

  desc 'Check a repos integrity'
  command :check do |cmd|
    cmd.switch %i[verify-data], desc: 'Also verify data (slow)'
    cmd.action do |_, opts, args|
      configs = Configuration.load_configs
      logger.debug opts

      args = configs.filter(&:default_exec).map(&:name) if args.empty?
      raise BackupError, 'No configs selected' if args.empty?

      targets = parse_args args
      TaskRunner.new(configs).check opts, targets

    rescue Configuration::ConfigError => e
      logger.error e.to_s
      exit! false
    rescue BackupError => e
      logger.error e.to_s
      exit! false
    end
  end

  on_error do |exc|
    puts "[!] #{exc}"

    # return false to skip default error handling
    false
  end

  # Non GLI methods

  def self.parse_args(args)
    backups = {}
    args.each do |arg|
      config, repos = arg.split '::'
      backups[config] = nil
      next if repos.nil?

      backups[config] = Set.new repos.split(',')
    end
    backups
  end
end
