# frozen_string_literal: true

require 'yaml'
require 'flood/logging'
require 'flood/utils'

# This module contains classes and methods related to reading the configuration
module Configuration
  class ConfigError < StandardError
  end

  REQUIRED_KEYS_GLOBAL = %w[name paths repos].freeze
  REQUIRED_KEYS_REPO = %w[name type compression encryption path archive-name
                          prune].freeze
  REPO_TYPES = %w[local ssh].freeze
  ENC_MODES = %w[authenticated repokey repokey-blake2].freeze
  CONF_DIR_USER = File.join Dir.home, '.config/flood/'
  CONF_DIR_SYSTEM = '/etc/flood'

  # Class holding the backup configuration.
  #
  # One of these will be constructed for each config file.
  class BackupConfig
    attr_reader :name, :paths, :excludes, :pre_commands, :repos, :default_exec

    def initialize(name, paths, excludes, pre_commands, repos, default_exec)
      @name = name
      @paths = paths
      @excludes = excludes || []
      @pre_commands = pre_commands || []
      @repos = repos
      @default_exec = default_exec
      @default_exec = true if default_exec.nil?
    end
  end

  # Class containing information for one backup target = repository
  class RepoConfig
    attr_reader :name, :type, :compression, :encryption, :passphrase, :host,
                :user, :path, :archive_name, :append_datetime, :prune, :keep,
                :corresponding_backup

    def initialize(conf_hash, backup_name)
      @name = conf_hash['name']
      @corresponding_backup = backup_name
      @type = conf_hash['type'].to_sym
      @compression = conf_hash['compression']
      @encryption = conf_hash['encryption']
      @passphrase = conf_hash['passphrase']
      @host = conf_hash['host']
      @user = conf_hash['user']
      @path = conf_hash['path']
      @archive_name = conf_hash['archive-name']
      @prune = conf_hash['prune']
      @keep = {}
      if @prune
        conf_hash['keep'].each do |key, val|
          accessor = case key
                     when 'daily'
                       :d
                     when 'weekly'
                       :w
                     when 'monthly'
                       :m
                     when 'yearly'
                       :y
                     end
          @keep[accessor] = val
        end
      end

      sane?
    end

    private

    def sane?
      if type == :ssh && (host.nil? || host.empty?)
        raise ConfigError,
              'repo type is remote, but there is no host specified'
      end
    end
  end

  # Class for looking up values in a ssh config
  class SshConfig
    include Logging

    attr_reader :entries

    def initialize(entries = [])
      @entries = {}
      entries.each do |entry|
        hosts = entry.delete :host
        hosts.each do |h|
          @entries[h] = entry
        end
      end
      logger.debug "New SshConfig: #{@entries}"
    end

    def get_hostname(host)
      entries[host]&.fetch(:hostname, nil)
    end

    def self.load_config
      config_path = File.join Dir.home, '.ssh/config'
      return SshConfig.new unless File.exist? config_path

      file_contents = File.read config_path
      sections = []
      current = nil
      file_contents.lines.each do |line|
        line = line.rstrip
        case line
        when /^Host\s/
          sections << current unless current.nil?
          current = {}
          current[:host] = line.split[1..]
        else
          next if line.empty?

          parts = line.strip.split
          key = parts.shift.downcase.to_sym
          current[key] = parts.shift
        end
      end
      sections << current unless current.nil?
      SshConfig.new sections
    end
  end

  def self.load_configs
    config_dir = if root_user?
                   CONF_DIR_SYSTEM
                 else
                   CONF_DIR_USER
                 end
    configs = []
    Dir.entries(config_dir).map { |e| File.join config_dir, e }
       .filter { |e| File.file?(e) && %w[.yaml .yml].include?(File.extname(e)) }
       .each do |f|
      configs << parse_config(File.read(f))
    rescue ConfigError => e
      Logging.logger.error "Could not load config file '#{f}': #{e}"
      next
    end
    raise BackupError, 'No valid configs available' if configs.empty?

    configs
  end

  def self.parse_config(config_file)
    conf_yaml = YAML.safe_load config_file
    missing_keys = REQUIRED_KEYS_GLOBAL.reject { |k| conf_yaml.include?(k) && !conf_yaml[k].nil? }
    raise ConfigError, "missing key(s) in config: #{missing_keys.join(',')}" unless missing_keys.empty?

    repos = conf_yaml['repos'].map { |rc| parse_repo rc, conf_yaml['name'] }
    BackupConfig.new conf_yaml['name'], conf_yaml['paths'], conf_yaml['excludes'],
                     conf_yaml['pre-commands'], repos, conf_yaml['default_exec']
  end

  def self.parse_repo(repo_conf, backup_name)
    missing_keys = REQUIRED_KEYS_REPO.reject { |k| repo_conf.include?(k) && !repo_conf[k].nil? }
    raise ConfigError, "missing key(s) in repo config: #{missing_keys.join(',')}" unless missing_keys.empty?

    if ENC_MODES.include?(repo_conf['encryption']) && repo_conf['passphrase'].nil?
      raise ConfigError,
            "encryption enabled but no passphrase specified in '#{repo_conf['name']}'"
    end

    raise ConfigError, "invalid repo type '#{repo_conf['type']}'" unless REPO_TYPES.include? repo_conf['type']

    raise ConfigError, 'prune set to true, but no keep config' if repo_conf['prune'] && repo_conf['keep'].nil?

    RepoConfig.new(repo_conf, backup_name)
  end
end
