# frozen_string_literal: true

require 'open3'
require 'English'

class CommandError < StandardError
end

class Command
  attr_reader :cmd, :args, :output_queue, :input_queue

  def initialize(cmd, args = [])
    @cmd = cmd
    @args = args
    @wait_thread = nil
    @started = false
    @stdout = []
    @stderr = []
    @stdout_new = []
    @stderr_new = []
    @output_queue = Queue.new
    @input_queue = Queue.new
    @output_mutex = Mutex.new
    @retval = nil
  end

  def start(with_queue: false)
    sin, sout, serr, wt = Open3.popen3(*cmd, *args)

    @wait_thread = wt
    @started = true

    Thread.new do
      until (line = sout.gets).nil?
        @output_mutex.synchronize do
          line = line.rstrip
          @stdout << line
          @stdout_new << line
        end
        @output_queue << [:stdout, line] if with_queue
      end
      sout.close
    end

    Thread.new do
      until (line = serr.gets).nil?
        @output_mutex.synchronize do
          line = line.rstrip
          @stderr << line
          @stderr_new << line
        end
        @output_queue << [:stderr, line] if with_queue
      end
      serr.close
    end

    if with_queue
      Thread.new do
        while (input = @input_queue.pop)
          sin.write input
        end
        sin.close
      end
    else
      sin.close
    end

    self
  end

  def kill
    Process.kill 'INT', wait_thread.pid if running?
    wait
  end

  def stdout_new
    @output_mutex.synchronize do
      @stdout_new.shift
    end
  end

  def stderr_new
    @output_mutex.synchronize do
      @stderr_new.shift
    end
  end

  def stdout
    @output_mutex.synchronize do
      @stdout
    end
  end

  def stderr
    @output_mutex.synchronize do
      @stderr
    end
  end

  def stdout_joined
    stdout.join("\n")
  end

  def stderr_joined
    stderr.join("\n")
  end

  def running?
    return false unless @started

    wait_thread.alive?
  end

  def wait
    return unless @started
    return unless @retval.nil?

    @retval = @wait_thread.value
    @output_queue.close
    @input_queue.clear
    @input_queue.close
  end

  def retval_thread
    Thread.new { retval }
  end

  def add_arg(arg)
    args << arg
  end

  def append_args(arg_arr)
    args.concat arg_arr
  end

  def retval
    return nil unless @started

    wait
    @retval
  end

  def success?
    retval&.success? || false
  end

  def to_s
    "<Cmd cmd=#{@cmd} args=#{@args}>"
  end

  private

  attr_accessor :wait_thread
  attr_writer :args, :retval
end
