# frozen_string_literal: true

require 'open3'
require 'English'

require 'flood/logging'
require 'flood/utils'
require 'flood/command'
require 'flood/log_parsing'

# This class handles the borg execution.
class Borg
  include Logging
  include MsgParser

  attr_reader :last_cmd

  def initialize(repo, passphrase)
    @repo = repo
    @passphrase = passphrase
    @last_cmd = nil
  end

  def archives
    set_env

    cmd = Command.new %w[borg], %w[list --json]
    @last_cmd = cmd
    cmd.start.wait
    raise BackupError, desc_error(cmd.stderr_joined) unless cmd.success?

    res = JSON.parse cmd.stdout_joined
    res['archives'].map { |a| a['archive'] }.sort
  end

  def info
    set_env

    cmd = Command.new %w[borg], %w[info --json]
    @last_cmd = cmd
    cmd.start.wait
    raise BackupError, desc_error(cmd.stderr_joined) unless cmd.success?

    JSON.parse cmd.stdout_joined
  end

  def check(verify_data: false)
    set_env
    logger.debug "verify_data: #{verify_data}"

    cmd = Command.new %w[borg], %w[check --log-json --progress -v]
    cmd.add_arg '--verify-data' if verify_data
    @last_cmd = cmd
    logger.debug "Prepared command: #{cmd}"

    cmd.start with_queue: true

    cmd_success = Thread.start { cmd.success? }
    yield cmd.output_queue
    cmd_success
  end

  def repo_available?(interactive: false)
    set_env

    cmd = Command.new(%w[borg], %w[info --log-json]).start with_queue: interactive
    retval_t = cmd.retval_thread
    if interactive
      while (msg = cmd.output_queue.pop)
        _, msg = msg
        msg = parse_logmsg msg
        next unless msg.is_a?(InputPrompt) && msg.msgid == 'BORG_RELOCATED_REPO_ACCESS_IS_OK'

        match = Regexes::RELOC.match msg.message
        logger.warn "The repo seems to have been relocated from #{match[:prev]} to #{match[:now]}"
        print '[?] Continue? (y/N) > '
        cmd.input_queue << $stdin.gets
      end
    end

    retval_t.value.success?
  end

  def init(enc_type)
    logger.debug "Initialising new repo. Using encryption type '#{enc_type}'"
    set_env

    logger.info "Initialising repo '#{repo}'"
    cmd = Command.new %w[borg], %W[init -e #{enc_type} #{repo}]
    @last_cmd = cmd
    cmd.start.wait
    raise BackupError, "Could not add repo:\n#{cmd.stderr_joined}" unless cmd.success?

    logger.info "New repo initialised at '#{repo}'"
    return unless enc_type == 'repokey-blake2'

    logger.info 'Using repokey-blake2 will lead to errors when using Borg version 1.0.8 and prior'
  end

  def backup(args, with_queue: false)
    set_env

    cmd = Command.new %w[borg], %w[create --json --log-json]
    cmd.append_args args
    logger.info 'Starting backup'
    @last_cmd = cmd
    retval = cmd.start(with_queue: with_queue).retval_thread
    logger.debug "Started borg command: #{cmd}"
    yield cmd.output_queue, cmd.input_queue if with_queue
    retval.value
  rescue Interrupt
    logger.info 'Interrupt received. Exiting'
    cmd.kill
    exit! true
  end

  def prune(keeps, &ofmt)
    set_env

    ergx = %r{(?<btype>Keeping archive|Pruning archive) \((?<ruleornum>rule: .*|\d+/\d+)\):\s+(?<bname>.*)\s+[A-Z][a-z]{2}, (?<bdt>\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}).*}
    archives_kept = []
    archives_pruned = []

    logger.info 'Pruning old backups'
    cmd = Command.new %w[borg], %w[prune --log-json --list]
    keeps.each { |k, v| cmd.append_args ["-#{k}", v.to_s] }
    logger.debug cmd

    retval = cmd.start(with_queue: true).retval_thread
    cmd.input_queue.close
    while (msg = cmd.output_queue.pop)
      type, line = msg
      next if type == :stdout

      msg = parse_logmsg(line)
      next unless msg.is_a?(LogMessage)

      match = ergx.match msg.message
      next if match.nil?

      case match[:btype]
      when /Keeping/
        archives_kept << match[:bname].strip
      when /Pruning/
        archives_pruned << match[:bname].strip
      end
    end

    logger.info 'Pruning finished'
    logger.error 'Error while pruning' unless retval.value.success?
    ofmt.call archives_pruned.sort, archives_kept.sort
  rescue Interrupt
    logger.info 'Interrupt received. Exiting'
    cmd.kill
    exit! true
  end

  def compact
    cmd = Command.new %w[borg], %w[compact --progress --log-json]
    cmd.add_arg repo
    @last_cmd = cmd

    cmd.start.wait

    raise CompactionError, 'Compaction not successful' unless cmd.success?
  end

  def mount(archive, mountpoint)
    set_env

    if archive.nil?
      logger.debug "borg mount #{repo} #{mountpoint} 2>&1"
      out = `borg mount #{repo} #{mountpoint} 2>&1`
    else
      logger.debug "borg mount '::#{archive}' #{mountpoint} 2>&1"
      out = `borg mount '::#{archive}' #{mountpoint} 2>&1`
    end
    raise BackupError, "Could not execute 'borg mount' (#{out.strip})" unless $CHILD_STATUS.success?
  end

  private

  attr_accessor :repo, :passphrase

  def set_env
    ENV['BORG_REPO'] = repo
    ENV['BORG_PASSPHRASE'] = passphrase
  end

  def desc_error(error_string)
    # TODO: add more descriptions
    case error_string
    when /resolve hostname/
      'unable to resolve hostname for repo, check availability'
    else
      error_string
    end
  end
end
