#! /usr/bin/env ruby
# frozen_string_literal: true

require 'time'
require 'yaml'
require 'json'
require 'io/console'

require 'flood/utils'
require 'flood/borg'
require 'flood/command'
require 'flood/log_parsing'

# A class for coordinating a backup to a single repo
class BackupGen
  include Logging
  include Formatting
  include RepoTools
  include MsgParser
  include FsUtils

  def initialize(local_conf, repo_conf)
    @paths = local_conf[:paths]
    @excludes = local_conf[:excludes]
    @local_conf = local_conf
    @repo_conf = repo_conf
    @src_size = 0
    @noterm = IO.console.nil?
    @ssh_conf = Configuration::SshConfig.load_config
    @term_width = if @noterm
                    256
                  else
                    self.class.term_width
                  end

    @run_start = nil
    Signal.trap('SIGWINCH', proc { @term_width = self.class.term_width }) unless @noterm
    full_repo_path = format_path repo_conf
    @borg = Borg.new full_repo_path, repo_conf.passphrase
  end

  def self.term_width
    Integer IO.console.winsize[1]
  end

  def run
    logger.debug 'Executing pre-commands'
    exec_pre_cmds
    logger.debug 'Starting backup'
    backup
    prune if repo_conf.prune
  end

  private

  attr_accessor :paths, :excludes, :repo_conf, :borg, :notify,
                :local_conf, :ssh_conf, :src_size

  def backup
    preflight_checks
    logger.debug 'Preflight checks passed'

    backup_name = gen_backup_name
    logger.debug "Backup name will be #{backup_name}"
    foreground = !local_conf[:background]

    raise BackupError, 'The target repo is not available' unless borg.repo_available? interactive: foreground
    raise BackupError, "Archive '#{backup_name}' already exists" if borg.archives.include? backup_name

    cond_dirs = eval_path_conds

    included_paths = cond_dirs.filter_map { |d, i| d if i }
    raise BackupError, 'No backup paths left after evaluating conditions' if included_paths.empty?

    skipped_paths = cond_dirs.filter_map { |d, i| d unless i }

    borg_args = foreground ? %w[--progress] : []
    borg_args += format_excludes
    borg_args << '-C' << repo_conf.compression
    borg_args << "::#{backup_name}"
    borg_args += included_paths

    logger.info "Backup name is '#{backup_name}'"
    print_path_details skipped_paths

    if foreground
      logger.debug 'Approximating source size'
      @src_size = get_approx_size included_paths, excludes
      logger.info "Approximate source size is #{format_size @src_size}"

      run_foreground borg_args
    else
      run_background borg_args
    end
  end

  def prune
    if local_conf[:background]
      borg.prune repo_conf.keep do |pruned, kept|
        logger.info "Pruned archives: #{pruned.length}, kept archives: #{kept.length}"
      end
    else
      borg.prune repo_conf.keep do |pruned, kept|
        logger.info 'Archives pruned:'
        pruned.each { |a| puts "  - #{a}" }
        logger.info 'Archives kept:'
        kept.each { |a| puts "  - #{a}" }
      end
      logger.info "Starting compaction for #{repo_conf.name} @ #{repo_conf.corresponding_backup}"
      borg.compact
      logger.info 'Compaction successful'
    end
  end

  def preflight_checks
    base_avail = -> { Dir.exist? File.dirname(repo_conf.path) }

    raise BackupError, 'No paths given.' if paths.empty?

    logger.debug 'Checking for repo availability'
    if repo_conf.type == :local
      logger.debug 'Repo type is local'
      unless base_avail.call
        logger.warn "Backup device does not seem to be available (yet). I'll retry in a bit."
        avail = (1..3).any? do
          sleep 60
          base_avail.call
        end
        raise BackupError, 'Backup device unavailable. Exiting.' unless avail
      end

      borg.init repo_conf.encryption unless Dir.exist? repo_conf.path
    else
      logger.debug 'Repo type is SSH'
      raise BackupError, "Host '#{repo_conf.host}' is not available" unless host_avail?

      unless borg.repo_available?
        logger.info 'Trying to create repo'
        borg.init repo_conf.encryption
      end
    end

    missing_paths = check_paths paths
    return if missing_paths.empty?

    missing_paths.each { |p| logger.error "Missing backup path: #{p}" }
    raise BackupError, "#{missing_paths.length} backup paths not available in filesystem"
  end

  def host_avail?
    logger.debug "SSH config: #{ssh_conf.entries}"
    host = ssh_conf.get_hostname(repo_conf.host) || repo_conf.host
    logger.debug "Trying to ping #{host}"
    Command.new(%w[ping], %W[-c1 #{host}]).start.success?
  end

  def exec_pre_cmds
    local_conf[:pre_commands].each do |cmd|
      logger.info "Exec '#{cmd}'"
      res = `#{cmd}`
      raise BackupError, "Command did not execute successfully: #{res}" unless $CHILD_STATUS.success?
    end
  end

  def eval_path_conds
    paths.map do |d|
      if d.include? ':'
        d, rule = d.split ':'
        d = File.expand_path d unless File.absolute_path? d
        [d, `pgrep -f #{rule}` == '']
      else
        d = File.expand_path d unless File.absolute_path? d
        [d, true]
      end
    end
  end

  def gen_backup_name
    "#{Time.now.strftime('%F %R')} #{repo_conf.archive_name}"
  end

  def format_progress(progress, previous)
    num_double_width = ->(str) { str.chars.filter { |c| c.bytes.length >= 3 }.length }

    @run_start = progress.time if @run_start.nil?
    backup_finished = progress.path.nil?
    if backup_finished
      orig = format_size previous.last.osize, ext_spacing: true
      comp = format_size previous.last.csize, ext_spacing: true
      dedup = format_size previous.last.dsize, ext_spacing: true
    else
      orig = format_size progress.osize, ext_spacing: true
      comp = format_size progress.csize, ext_spacing: true
      dedup = format_size progress.dsize, ext_spacing: true
    end

    time_passed = format_dur(progress.time.to_f - @run_start)
    percent_done = (progress.osize.to_f / src_size).clamp(0.0, 1.0)
    percent_done = 1.0 if backup_finished

    # Elapsed time, current file
    rem_time = if previous.empty?
                 'N/A'
               elsif backup_finished
                 '00:00:00'
               else
                 format_dur calc_remaining(progress, previous)
               end
    time_file = +''
    time_file << format('ET: %<passed>11s | RT: %<rem>11s | File: ', passed: time_passed, rem: rem_time)
    filepath = progress.path || ''
    dw = num_double_width.call filepath
    flen = filepath.length + dw
    if flen + 2 > @term_width - time_file.length
      ml = @term_width - time_file.length - 2 - dw
      sel = ml / 2 - 2

      filepath = "#{filepath[0..sel]}...#{filepath[-sel..]}"
    end
    time_file << filepath.to_s

    # Percent done, size information
    proc_bar_len = @term_width / 3 - 10
    filled = (percent_done * proc_bar_len).floor
    progress_size = +''
    progress_size << '['
    progress_size << '=' * filled
    progress_size << ' ' * (proc_bar_len - filled)
    progress_size << format('|%<pdone>6.2f%%] ', pdone: percent_done * 100)
    progress_size << "#{orig} / #{format_size src_size} | "
    progress_size << "C: #{comp} | "
    progress_size << "D: #{dedup}"
    "#{progress_size}\n#{time_file}"
  end

  def print_stats(stats)
    archive = stats['archive']
    osize = Float archive['stats']['original_size']
    dsize = Float archive['stats']['deduplicated_size']
    lim = archive['limits']['max_archive_size']
    lim *= 100
    duration = format_dur archive['duration']

    puts ''
    puts " Repo name:         #{repo_conf.name}"
    puts " Archive name:      #{archive['name']}"
    puts " Backup duration:   #{duration}"
    puts " Files backed up:   #{archive['stats']['nfiles']}"
    puts " Original size:     #{format_size osize}"
    puts " Deduplicated size: #{format_size dsize}"
    puts " Archive limit:     #{lim.round(3)}%"
    puts ''
  end

  def print_path_details(skipped)
    if local_conf[:background]
      logger.info "Skipped paths: #{skipped}" unless skipped.empty?
    else
      unless skipped.empty?
        logger.info 'Paths skipped:'
        skipped.each { |path| puts "  - #{path}" }
      end
    end
  end

  def run_background(args)
    retval = borg.backup(args)
    raise BackupError, "Backup completed with errors. Error code: #{retval.exitstatus}" unless retval.success?

    archive = JSON.parse(borg.last_cmd.stdout_joined)['archive']
    stats = archive['stats']
    limit = archive['limits']['max_archive_size'] * 100
    logger.info "Backup finished after #{format_dur archive['duration']}"
    logger.info "Backup includes #{stats['nfiles']} files"
    logger.info "Size stats => Orig: [#{format_size stats['original_size']}] "\
                "Comp: [#{format_size stats['compressed_size']}] "\
                "Dedup: [#{format_size stats['deduplicated_size']}] Lim: #{limit.round 3}%"
  rescue JSON::ParserError
    logger.warn 'Backup seems to be fine but I couldn\'t decode the output'
  end

  def run_foreground(args)
    retval = borg.backup args, with_queue: true do |outq, inq|
      prog_drawn = false
      prev_progress = []
      while (msg = outq.pop)
        type, line = msg
        next unless type == :stderr

        msg = parse_logmsg line
        if msg.is_a?(ArchiveProgress)
          if !prog_drawn
            prog_drawn = true
          else
            clear_line
            IO.console.cursor_up 1
            clear_line
          end
          print format_progress(msg, prev_progress)
          $stdout.flush
          prev_progress << msg
          prev_progress.shift if prev_progress.length > 200
        elsif msg.is_a?(InputPrompt)
          puts msg.message
          print ':> '
          inq << $stdin.gets
        end
      end
    end
    unless retval.success?
      puts ''
      logger.error "Backup completed with errors. Error code: #{retval.exitstatus}"
      raise BackupError, 'Backup completed with errors.'
    end
    clear_line
    logger.info 'Backup completed successfully'
    stats = JSON.parse borg.last_cmd.stdout_joined
    print_stats stats
  rescue JSON::ParserError
    logger.warn 'Backup seems to be fine but I couldn\'t decode the output'
  end

  def format_excludes
    excludes.map { |e| ['-e', e] }.flatten
  end

  def calc_remaining(current, previous)
    entries = previous + [current]
    avg_dr = pairs(entries).map do |fst, snd|
      data_processed = snd.osize - fst.osize
      time_elapsed = snd.time - fst.time
      data_processed.to_f / time_elapsed
    end.sum / (entries.length - 1)
    (src_size - current.osize) / avg_dr
  end
end
